{*> Ver: V1.1 *********      History      ***************************\

V1.0  09/13/2009  Released
V1.1  09/20/2009  Redesigned to use basie class.
                  Nuber of small fixes.

Legal issues: Copyright (C) 2009 by Boian Mitov
              mitov@mitov.com
              www.mitov.com

              This software is provided 'as-is', without any express or
              implied warranty.  In no event will the author be held liable
              for any  damages arising from the use of this software.

              Permission is granted to anyone to use this software for any
              purpose, including commercial applications, and to alter it
              and redistribute it freely, subject to the following
              restrictions:

              1. The origin of this software must not be misrepresented,
                 you must not claim that you wrote the original software.
                 If you use this software in a product, an acknowledgment
                 in the product documentation would be appreciated but is
                 not required.

              2. Altered source versions must be plainly marked as such, and
                 must not be misrepresented as being the original software.

              3. This notice may not be removed or altered from any source
                 distribution.
\***************************************************************************}

unit BitmapVisualizer;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, BasicExternalViewerVisualizer, ExtCtrls, StdCtrls;

type
  TBitmapViewerFrame = class(TBasicVisualizerViewerFrame)
    Panel1: TPanel;
    Label1: TLabel;
    Label2: TLabel;
    SizeLabel: TLabel;
    FormatLabel: TLabel;
    StretchedCheckBox: TCheckBox;
    ProportionalCheckBox: TCheckBox;
    MousePanel: TPanel;
    ColorShape: TShape;
    ColorLabel: TLabel;
    MouseLabel: TLabel;
    PreviewScrollBox: TScrollBox;
    PreviewImage: TImage;
    procedure StretchedCheckBoxClick(Sender: TObject);
    procedure ProportionalCheckBoxClick(Sender: TObject);
    procedure PreviewImageMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure PreviewImageMouseEnter(Sender: TObject);
    procedure PreviewImageMouseLeave(Sender: TObject);
  protected
    procedure EvaluateExpression( Context: TVisualizerDebugContext; const Expression, TypeName, EvalResult: string); override;

  end;

procedure Register;

implementation

{$R *.dfm}

uses
  TypInfo, ToolsAPI;

resourcestring
  sBitmapVisualizerName = 'TBitmap Visualizer for Delphi';
  sBitmapVisualizerDescription = 'Displays the content of the actual image held in a TBitmap instance';
  sBitmapMenuText = 'Show Bitmap';
  sBitmapFormCaption = 'TBitmap Visualizer for %s';
  sBitmapCantDisplay = 'can''t display';

type
  TBitmapDebuggerVisualizer = class(TBasicDebuggerVisualizer)
  protected
    function CreateForm( const Expression : String ) : TInterfacedObject; override;

  public
    function GetMenuText() : string;  override;
    function GetVisualizerName() : string;  override;
    function GetVisualizerDescription() : string;  override;

  public
    constructor Create();

  end;

var
  SLBlockVisualizer: IOTADebuggerVisualizer;

constructor TBitmapDebuggerVisualizer.Create();
begin
  inherited;
  AddType( 'TBitmap' );

  AddType( 'Graphics::TBitmap *' );
  AddType( 'const Graphics::TBitmap *' );
  AddType( 'const Graphics::TBitmap' );
  AddType( 'Graphics::TBitmap' );
end;

function TBitmapDebuggerVisualizer.CreateForm( const Expression : String ) : TInterfacedObject;
begin
  Result := TBasicDebuggerVisualizerForm.Create( Expression, TBitmapViewerFrame, 'BitmapDebugVisualizer', sBitmapFormCaption );
end;

function TBitmapDebuggerVisualizer.GetVisualizerDescription() : string;
begin
  Result := sBitmapVisualizerDescription;
end;

function TBitmapDebuggerVisualizer.GetMenuText() : string;
begin
  Result := sBitmapMenuText;
end;

function TBitmapDebuggerVisualizer.GetVisualizerName() : string;
begin
  Result := sBitmapVisualizerName;
end;

procedure TBitmapViewerFrame.StretchedCheckBoxClick(Sender: TObject);
begin
  PreviewImage.Stretch := StretchedCheckBox.Checked;
  if( StretchedCheckBox.Checked ) then
    PreviewImage.Align := alClient

  else
    PreviewImage.Align := alNone;

end;

procedure TBitmapViewerFrame.ProportionalCheckBoxClick(Sender: TObject);
begin
  PreviewImage.Proportional := ProportionalCheckBox.Checked;
end;

procedure TBitmapViewerFrame.PreviewImageMouseEnter(Sender: TObject);
begin
  MousePanel.Visible := True;
end;

procedure TBitmapViewerFrame.PreviewImageMouseLeave(Sender: TObject);
begin
  MousePanel.Visible := False;
end;

procedure TBitmapViewerFrame.PreviewImageMouseMove(Sender: TObject;
  Shift: TShiftState; X, Y: Integer);
var
  XCoef : Real;
  YCoef : Real;

begin
  XCoef := 1;
  YCoef := 1;
  if( PreviewImage.Stretch ) then
    if( PreviewImage.Proportional ) then
      begin
      if( PreviewImage.Picture.Width / PreviewImage.Picture.Height > PreviewImage.Width / PreviewImage.Height ) then
        begin
        XCoef := PreviewImage.Picture.Width / PreviewImage.Width;
        YCoef := XCoef;
        end

      else
        begin
        YCoef := PreviewImage.Picture.Height / PreviewImage.Height;
        XCoef := YCoef;
        end;

      end

    else
      begin
      XCoef := PreviewImage.Picture.Width / PreviewImage.Width;
      YCoef := PreviewImage.Picture.Height / PreviewImage.Height;
      end;

  X := Trunc( X * XCoef );
  Y := Trunc( Y * YCoef );
  MouseLabel.Caption := IntToStr( X ) + ':' + IntToStr( Y );
  if( ( X > PreviewImage.Picture.Width ) or ( Y > PreviewImage.Picture.Height ) ) then
    MousePanel.Visible := False

  else
    begin
    MousePanel.Visible := True;
    ColorShape.Brush.Color := PreviewImage.Canvas.Pixels[ X, Y ];
    ColorLabel.Caption := ColorToString( PreviewImage.Canvas.Pixels[ X, Y ] );
    end;

end;

procedure EnumTypeToStrings( EnumInfo : PTypeInfo; Values : TStrings );
var
  I             : Integer;
  ConditionName : String;
  TypeData      : PTypeData;

begin
  TypeData := GetTypeData(EnumInfo);

  Values.Clear();
  for I := 0 to TypeData.MaxValue do
    begin
    ConditionName := GetEnumName( EnumInfo, I );
    Values.Add( ConditionName );
    end;

end;
//------------------------------------------------------------------------------
function EnumIndexFromIdent( EnumInfo : PTypeInfo; AIdent : String ) : Integer;
var
  Values : TStringList;

begin
  Values := TStringList.Create();
  EnumTypeToStrings( EnumInfo, Values );
  Result := Values.IndexOf( AIdent );
  if( Result < 0 ) then
    Result := 0;

  Values.Free();
end;
//---------------------------------------------------------------------------

procedure TBitmapViewerFrame.EvaluateExpression( Context: TVisualizerDebugContext; const Expression, TypeName, EvalResult: string );
const
  BitCounts: array [pf1Bit..pf32Bit] of Byte = (1,4,8,16,16,24,32);

var
  ASize       : TSize;
  APixelFormat: TPixelFormat;
  ARemoteHandle : LongWord;
  BI: TBitmapInfoHeader;
  AScanSize   : Integer;
  I           : Integer;
  ARemoteMem1    : LongWord;
  ARemoteMem2    : LongWord;
  ASeparator1    : String;
  ASeparator2    : String;
  AIsCPP         : Boolean;

begin
  if( Pos( '*', TypeName ) > 0 ) then
    begin
    ASeparator1 := '( ';
    ASeparator2 := ' )->';
    AIsCPP := True;
    end

  else
    begin
    ASeparator1 := '';
    ASeparator2 := '.';
    AIsCPP := False;
    end;

  ASize.cx := EvaluateIntegerValue( Context, ASeparator1 + Expression + ASeparator2 + 'Width' );
  ASize.cy := EvaluateIntegerValue( Context, ASeparator1 + Expression + ASeparator2 + 'Height' );
  SizeLabel.Caption := IntToStr( ASize.cx ) + 'x' + IntToStr( ASize.cy );
  FormatLabel.Caption := EvaluateStringValue( Context, ASeparator1 + Expression + ASeparator2 + 'PixelFormat' );

  APixelFormat := TPixelFormat( EnumIndexFromIdent( TypeInfo( TPixelFormat ), FormatLabel.Caption ));
  if( ( APixelFormat = pfDevice ) or ( APixelFormat = pfCustom )) then
    begin
    PreviewImage.Visible := False;
    MessageLabel.Visible := True;
    MessageLabel.Font.Size := 20;
    MessageLabel.Caption := sBitmapCantDisplay;
    Exit;
    end;

  PreviewImage.AutoSize := True;
  PreviewImage.Picture.Bitmap.Width := ASize.cx;
  PreviewImage.Picture.Bitmap.Height := ASize.cy;

  if( APixelFormat = pf15bit ) then
    APixelFormat := pf16bit;

  PreviewImage.Picture.Bitmap.PixelFormat := APixelFormat;

  PreviewImage.Picture.Bitmap.AlphaFormat := TAlphaFormat( EnumIndexFromIdent( TypeInfo( TPixelFormat ), EvaluateStringValue( Context, ASeparator1 + Expression + ASeparator2 + 'AlphaFormat' )));

  ARemoteHandle := 0;
  while( ARemoteHandle = 0 ) do
    begin
    ARemoteHandle := EvaluateLongWordValue( Context, 'CreateCompatibleDC( 0 )' );
    Context.DebugSvcs.ProcessDebugEvents();
    end;

  if( ARemoteHandle <> 0 ) then
    begin
//          ARemoteMem := EvaluateLongWordValue( 'GetMemory(' + IntToStr( Image1.Picture.Bitmap.Width * Image1.Picture.Bitmap.Height ) + ')' );
    BI.biSize := SizeOf(BI);
    BI.biWidth := PreviewImage.Picture.Bitmap.Width;
    BI.biHeight := PreviewImage.Picture.Bitmap.Height;
    BI.biBitCount := BitCounts[ PreviewImage.Picture.Bitmap.PixelFormat ];

    BI.biClrUsed := 0;
    BI.biPlanes := 1;
    AScanSize := BytesPerScanLine(BI.biWidth, BI.biBitCount, 32 );
    BI.biSizeImage := AScanSize * Abs(BI.biHeight);


    BI.biClrImportant := 0;

    for I := 0 to 100 do
      begin
      ARemoteMem1 := AllocateRemoteMem( Context, SizeOf( TBitmapInfoHeader ));
      if( ARemoteMem1 <> 0 ) then
        Break;

      Context.DebugSvcs.ProcessDebugEvents();
      end;

    if( ARemoteMem1 <> 0 ) then
      begin
      for I := 0 to 100 do
        begin
        ARemoteMem2 := AllocateRemoteMem( Context, AScanSize * PreviewImage.Picture.Bitmap.Height );
        if( ARemoteMem2 <> 0 ) then
          Break;

        Context.DebugSvcs.ProcessDebugEvents;
        end;

      if( ARemoteMem2 <> 0 ) then
        begin
        Context.CurProcess.WriteProcessMemory( ARemoteMem1, SizeOf( BI ), BI );

        if( AIsCPP ) then
          EvaluateLongWordValue( Context, 'GetDIBits(' + IntToStr( ARemoteHandle ) + ', (' + Expression + ')->Handle, 0, ' + IntToStr( PreviewImage.Picture.Bitmap.Height ) + ', (void *)(' + IntToStr( ARemoteMem2 ) + ' ), ((TBitmapInfo *)' + IntToStr( ARemoteMem1 ) + ' ), DIB_RGB_COLORS )' )

        else
          EvaluateLongWordValue( Context, 'GetDIBits(' + IntToStr( ARemoteHandle ) + ', ' + Expression + '.Handle, 0, ' + IntToStr( PreviewImage.Picture.Bitmap.Height ) + ', Pointer(' + IntToStr( ARemoteMem2 ) + ' ), PBitmapInfo( ' + IntToStr( ARemoteMem1 ) + ' )^, DIB_RGB_COLORS )' );

        Context.CurProcess.ReadProcessMemory( ARemoteMem2, AScanSize * PreviewImage.Picture.Bitmap.Height, PreviewImage.Picture.Bitmap.ScanLine[ PreviewImage.Picture.Bitmap.Height - 1 ]^ );
        FreeRemoteMem( Context, ARemoteMem2 );
        end;

      FreeRemoteMem( Context, ARemoteMem1 );
      end;

    EvaluateLongWordValue( Context, 'DeleteDC(' + IntToStr( ARemoteHandle ) + ')' );
    end;

end;

procedure Register;
begin
  SLBlockVisualizer := TBitmapDebuggerVisualizer.Create();
  (BorlandIDEServices as IOTADebuggerServices).RegisterDebugVisualizer(SLBlockVisualizer);
end;

procedure RemoveVisualizer();
var
  DebuggerServices: IOTADebuggerServices;

begin
  if( Supports( BorlandIDEServices, IOTADebuggerServices, DebuggerServices ) ) then
    begin
    DebuggerServices.UnregisterDebugVisualizer(SLBlockVisualizer);
    SLBlockVisualizer := nil;
    end;

end;

initialization
finalization
  RemoveVisualizer();

end.
